--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Markets
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

require("sim/building.lua")

-- Market class definition, derived from LBuilding
LMarket = LBuilding:subclass
{
	__tostring = function(t) return "{Market:" .. tostring(t.name) .. "}" end,
	_Markets = {},
	
	ambient = "sfx/marketplace_loop.ogg",
}

-------------------------------------------------------------------------------
-- Functions for data description

function Market(t) return LMarket:new(t) end

-------------------------------------------------------------------------------
-- "constructor"

function LMarket:new(t)
	t = LBuilding.new(self, t)
	t.inventory = t.inventory or {}
	table.insert(self._Markets, t)
	return t
end

function LMarket:SetPort(p)
	LBuilding.SetPort(self, p)
	p.hasMarket = true
end

-------------------------------------------------------------------------------
-- Convenient data access

function LMarket:AllMarkets()
	return bsgArrayIterator(self._Markets)
end

-------------------------------------------------------------------------------
-- Building interaction

function LMarket:SelectCharacterFreePlay() return self.character[1] end

function LMarket:OnActivate()
	gActiveBuilding = self
	self:SetAmbient()
	local char,quest = self:SelectCharacter()

	-- Tutorial Special: During tutorial (rank 0) disable plantation in San Jose
	if gSim.rank == 0 and self.name == "sj_farm" and gSim.mode ~= "free" then
		local char = self.character[1]
		local message = GetString("backtotask")
		DisplayDialog { "ui/chartalk.lua", char=char, fullbody=message, ok="ok" }
	else
		local items = {}
		for name,_ in pairs(self.inventory) do
			local item = LItem:ByName(name)
			if item then table.insert(items, item) end
		end
		
		-- Truffle powder available wherever cacao is available after rank 3
		if gSim.rank >= 3 and self.inventory.cacao then
			table.insert(items, LItem:ByName("powder"))
		end
		
		DisplayDialog { "ui/marketui.lua", market=self, char=char, items=items }
	end
	
	self:ExitBuilding()
end

-------------------------------------------------------------------------------
-- Haggling

function LMarket:ComputeHaggle(c)
	local seasonWeek = bsgMod(gSim.weeks, 52)

	-- The "reasonableness" of a current price is a measure of where the
	-- actual price lies between the item low and high prices
	local count = 0
	local R = 0
-- HAGGLE based on gShopItems, not self.inventory -- truffle powder is added dynamically
--	for name,_ in pairs(self.inventory) do
--		local item = LItem:ByName(name)
	for _,item in pairs(gShopItems) do
		-- LOWER prices are better
		local target = (item.price - item.low) / (item.high - item.low)
		if not item:InSeason(seasonWeek) then
			-- Out-of-season pricing between 1.5x and 2x high
			target = (item.price - 1.5*item.high) / (2*item.high - 1.5*item.high)
		end
		R = R + target
		count = count + 1
	end
	-- The overall "reasonableness" of prices is the average reasonableness
	R = 100 * R / count
	
	-- Factor in the "haggle factor" for this character -- just a multiplier for R
	-- Lower haggleFactor means lower R, means less likely to haggle
	if c.haggleFactor then R = R * c.haggleFactor end

	-- Adjust according to character mood -- 10% bump in either direction
	if c:IsAngry() then R = R * .9
	elseif c:IsHappy() then R = R * 1.1
	end
	
	-- Now on a scale of 1..100, the lower value of R, the better the prices...
	-- If we roll lower than R, we can get better prices -- so as prices get
	-- better, our chances of rolling even lower are slim
	local H = bsgRandom(1,100)
	
	-- Roll is 20 pts under, prices drop
	-- 30 pts or more over, prices rise
	-- SO... if actual price is half way between low and high, R=50
	--  R=100: 80% drop, 20% stay the same
	--  R=50: 30% drop, 50% stay the same, 20% go up
	--  R=30: 10% dop, 50% stay the same, 40% go up
	--  R=0: 0% drop, 30% stay the same, 70% go up
	
	-- Force a successful haggle based on quest variable setting
	if LQuest:Variable("HaggleBonus") > 0 then
		H = R - 21
		LQuest._Variables.HaggleBonus = LQuest._Variables.HaggleBonus - 1
	end

	local response = "none"
	if H < R-20 then
		-- Prices drop toward minimum
		response = "down"
--		for name,_ in pairs(self.inventory) do
--			local item = LItem:ByName(name)
		for _,item in pairs(gShopItems) do
			local newPrice = bsgFloor((item.price + item.low) / 2)
			if not item:InSeason(seasonWeek) then
				-- Out-of-season minimum is 1.5x high
				newPrice = bsgFloor((item.price + 1.5*item.high) / 2)
			end
			item.price = newPrice
		end
		-- "Good" sound
		PlaySound("haggle_price_down")
	elseif H > R+30 then
		-- Prices raise to maximum, character becomes angry
		response = "up"
--		for name,_ in pairs(self.inventory) do
--			local item = LItem:ByName(name)
		for _,item in pairs(gShopItems) do
			item.price = item.high
			if not item:InSeason(seasonWeek) then
				-- Out-of-season minimum is 2x high
				item.price = 2*item.high
			end
		end
		c:SetMood(LCharacter.kAngry - 10)
		-- "Bad" sound
		PlaySound("haggle_price_up")
	end
	
	return response
end
